/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/


#ifdef BEST_FIRMWARE
This module must not be linked to firmware

#else
#include <typedefs.h>
#include "pcicode.h"

#endif


#include <confscan.h>
#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <regx20.h>
#include <regx21.h>
#include <session.h>


#define INCR_BUFFER	s += k, l += k


#define FOUR_NULL	NULL, NULL, NULL, NULL

static char *cscan_result [MAXHANDLES + 1] = {
  FOUR_NULL, FOUR_NULL, FOUR_NULL, FOUR_NULL, 
  FOUR_NULL, FOUR_NULL, FOUR_NULL, FOUR_NULL,
  FOUR_NULL, FOUR_NULL, FOUR_NULL, FOUR_NULL, 
  FOUR_NULL, FOUR_NULL, FOUR_NULL, FOUR_NULL,
  FOUR_NULL, FOUR_NULL, FOUR_NULL, FOUR_NULL, 
  FOUR_NULL, FOUR_NULL, FOUR_NULL, FOUR_NULL,
  FOUR_NULL, FOUR_NULL, FOUR_NULL, FOUR_NULL, 
  FOUR_NULL, FOUR_NULL, FOUR_NULL, FOUR_NULL,
  NULL
};


static void cs_vendor_device (b_int32 id,
			      b_ccharptrtype *vendor,
			      b_ccharptrtype *device)
{
  PPCI_VENTABLE	v;
  PPCI_DEVTABLE	d;
  static char  *unknown_vendor = "unknown vendor";
  static char  *unknown_device = "unknown device";
  static char   device_buffer [256];

  *vendor = unknown_vendor;
  *device = unknown_device;
  
  for (v = PciVenTable; v -> VenId != 0xffff; ++v) {
    if (v -> VenId == (id & 0x0000ffff)) {
      *vendor = v -> VenFull;
      goto device_label;
    }
  }

  return;

device_label:
  for (d = PciDevTable; d -> VenId != 0xffff; ++d) {
    if ((d -> VenId == (id & 0x0000ffff)) &&
	(d -> DevId == ((id & 0xffff0000) >> 16L))) {
      sprintf (device_buffer, "%s %s", d -> Chip, d -> ChipDesc);
      *device = device_buffer;
      break;
    }
  }
}


static void cs_classcode (b_int32 reg, b_ccharptrtype *classcode)
{
  switch (reg & 0xff000000) {
   case 0x00000000: 
    *classcode = "Backward Compatible";
    break;
	  
   case 0x01000000: 
    *classcode = "Mass Storage Controller";
    break;
	  
   case 0x02000000: 
    *classcode = "Network Controller"; 
    break;
	  
   case 0x03000000: 
    *classcode = "Display Controller"; 
    break;
    
   case 0x04000000: 
    *classcode = "Multimedia Device"; 
    break;
    
   case 0x05000000: 
    *classcode = "Memory Controller"; 
    break;
    
   case 0x06000000: 
    *classcode = "Bridge Device"; 
    break;
    
   case 0x07000000: 
    *classcode = "Communications Controller"; 
    break;
    
   case 0x08000000: 
    *classcode = "Generic System Peripheral"; 
    break;
    
   case 0x09000000: 
    *classcode = "Input Device"; 
    break;
    
   case 0x0a000000: 
    *classcode = "Docking Station"; 
    break;
    
   case 0x0b000000: 
    *classcode = "Processor"; 
    break;
    
   case 0x0c000000: 
    *classcode = "Serial Bus Controller"; 
    break;
    
   default:   
    *classcode = "unknown Class Code"; 
    break;
  }
}


static int cs_newline (b_charptrtype s)
{
  strcpy (s, "\n");
  return 1;
}


#define CS_FIELDSIZE    "%-24s"
#define CS_FIELD	CS_FIELDSIZE "- "

static int cs_format (b_charptrtype s,
		      b_ccharptrtype field, b_int32 value, b_int32 width)
{
  char *f,
        g [80];

  switch (width) {
  case 0:
    if (field) {
      f = CS_FIELD;
    }
    
    else {
      field = "";
      f = CS_FIELDSIZE;
    }
    
    break;
    
  case 4:
    f = CS_FIELD "%04x\\h";
    break;
    
  case 6:
    f = CS_FIELD "%06x\\h";
    break;

  case 8:
    f = CS_FIELD "%08x\\h";
    break;

  default:
    f = CS_FIELD "%02x\\h";
    break;
  }

  sprintf (g, f, field, value);
  return sprintf (s, "%-38s", g);
}


static int cs_command (b_charptrtype s, b_int32 value)
{
  int	k, l = 0;
  char	flag [10], flags [80];

  flags [0] = '\0';

  k = cs_format (s, "Command Register", value, 4);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;
  k = sprintf (s, "          ");
  INCR_BUFFER;

  sprintf (flag, "%cio ", value & (1 << 0) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cmem ", value & (1 << 1) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cma ", value & (1 << 2) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cspcl ", value & (1 << 3) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cmwi ", value & (1 << 4) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cvga ", value & (1 << 5) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cperr ", value & (1 << 6) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cwait ", value & (1 << 7) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cserr ", value & (1 << 8) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cfb2b ", value & (1 << 9) ? '+' : '-');
  strcat (flags, flag);
  
  k = sprintf (s, "%s", flags);
  INCR_BUFFER;

  return l;
}


static int cs_bridge_control (b_charptrtype s, b_int32 value)
{
  int	k, l = 0;
  char	flag [10], flags [80];

  flags [0] = '\0';

  k = cs_format (s, "Bridge Control Register", value, 4);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;
  k = sprintf (s, "          ");
  INCR_BUFFER;

  sprintf (flag, "%cperr ", value & (1 << 0) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cserr ", value & (1 << 1) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cisa ", value & (1 << 2) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cvga ", value & (1 << 3) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cmam ", value & (1 << 5) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%crst ", value & (1 << 6) ? '+' : '-');
  strcat (flags, flag);
  sprintf (flag, "%cfb2b ", value & (1 << 7) ? '+' : '-');
  strcat (flags, flag);
  
  k = sprintf (s, "%s", flags);
  INCR_BUFFER;

  return l;
}


static int cs_status (b_charptrtype s, b_ccharptrtype field, b_int32 value)
{
  int	k, l = 0;
  char	flag [10], flags [80], *v;

  flags [0] = '\0';

  k = cs_format (s, field, value, 4);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;
  k = sprintf (s, "          ");
  INCR_BUFFER;

  sprintf (flag, "%c66 ", value & (1 << 5) ? '+' : '-');
  strcat (flags, flag);
  
  sprintf (flag, "%cudf ", value & (1 << 6) ? '+' : '-');
  strcat (flags, flag);
  
  sprintf (flag, "%cfb2b ", value & (1 << 7) ? '+' : '-');
  strcat (flags, flag);
  
  sprintf (flag, "%cperr ", value & (1 << 8) ? '+' : '-');
  strcat (flags, flag);

  switch (value & 0x0600) {
  case 0x0000:
    v = "fast ";
    break;

  case 0x0200:
    v = "medium ";
    break;

  case 0x0400:
    v = "slow ";
    break;

  default:
    v = "subtr ";
    break;
  }

  strcat (flags, v);

  sprintf (flag, "%cst ", value & (1 << 11) ? '+' : '-');
  strcat (flags, flag);
  
  sprintf (flag, "%crt ", value & (1 << 12) ? '+' : '-');
  strcat (flags, flag);
  
  sprintf (flag, "%crm ", value & (1 << 13) ? '+' : '-');
  strcat (flags, flag);
  
  sprintf (flag, "%cserr ", value & (1 << 14) ? '+' : '-');
  strcat (flags, flag);
  
  sprintf (flag, "%cpar ", value & (1 << 15) ? '+' : '-');
  strcat (flags, flag);
  
  k = sprintf (s, "%s", flags);
  INCR_BUFFER;

  return l;
}


static int cs_baseaddr (b_charptrtype s, b_int32 n, b_int32 value)
{
  b_int32 mask;
  int	k, l = 0;
  char	field	[24];
  char *space, *prefetch;

  mask = 0xfffffff0;

  if (value & 0x00000001)
    mask |= 0x0000000c;
  
  sprintf (field, "Base Address %d", n);
  k = sprintf (s, CS_FIELD "        %08x\\h  ", field, value & mask);
  INCR_BUFFER;

  if (value & 0x00000001) {
    k = sprintf (s, "IO decoder");
  }

  else {
    switch (value & 0x00000006) {
    case 0x00:
      space = "32-bit";
      break;

    case 0x02:
      space = "1 Meg";
      break;

    case 0x04:
      space = "64-bit";
      break;

    case 0x06:
      space = "reserved";
      break;

    default:
      space = "illegal case";
      break;
    }

    prefetch = (value & 0x00000008 ? " prefetchable" : "");
    k = sprintf (s, "Memory Decoder %s%s",
		 space, prefetch);
  }

  INCR_BUFFER;

  k = cs_newline (s);
  INCR_BUFFER;
  return l;
}


static int cs_baseaddr64 (b_charptrtype s, b_int32 n,
			b_int32 value, b_int32 value64)
{
  int	k, l = 0;
  char	field	[24];
  char *prefetch;

  if (value & 0x00000001) {
  addr32_label:
    k = cs_baseaddr (s, n, value);
    INCR_BUFFER;
    k = cs_baseaddr (s, n + 1, value64);
    INCR_BUFFER;
  }

  else {
    switch (value & 0x00000006) {
    case 0x04:
      prefetch = (value & 0x00000008 ? " prefetchable" : "");
      sprintf (field, "Base Address %d", n);
      k = sprintf (s, CS_FIELD "%08x%08x\\h  ", field,
	       value64, value & 0xfffffff0);
      INCR_BUFFER;
      k = sprintf (s, "Memory Decoder 64-bit%s", prefetch);
      INCR_BUFFER;

      k = cs_newline (s);
      INCR_BUFFER;
      break;

    default:
      goto addr32_label;  
      break;
    }
  }

  return l;
}


static int cs_xromaddr (b_charptrtype s, b_int32 value)
{
  int	k, l = 0;
  
  k = cs_format (s, "Expansion ROM Address", value & 0xfffffffe, 8);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;
  return l;
}

  
static int cs_mapped_io (b_charptrtype s, b_int32 base, b_int32 limit)
{
  int	k, l = 0;
  
  k = sprintf (s, CS_FIELD, "Mapped I/O Filter");
  INCR_BUFFER;

  if ((base | limit) & 0x000f) {
    k = sprintf (s, "erratic base (%04x\\h) or limit (%04x\\h) value",
		 base, limit);
  }

  else {
    k = sprintf (s, "%08x\\h thru %08x\\h",
		 (base << 16L), (limit << 16L) | 0x000fffff);
  }

  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;
  return l;
}


static int cs_io (b_charptrtype s,
		  b_int32 base, b_int32 base16, b_int32 limit, b_int32 limit16)
{
  int	k, l = 0;

  limit16; /* not used */
  
  k = sprintf (s, CS_FIELD, "I/O Filter");
  INCR_BUFFER;

  switch (base & 0x0f) {
  case 0:
    /* 16-bit I/O */
    k = sprintf (s, "%04x\\h thru %04x\\h",
		 (base & 0xf0) << 8L, (limit & 0xf0) << 8L | 0x0fff);
    break;
    
  case 1:
    /* 32-bit I/O */
    k = sprintf (s, "%08x\\h thru %08x\\h",
		 ((base & 0xf0) << 8L) | (base16 << 16L),
		 ((limit & 0xf0) << 8L) | (base16 << 16L) | 0x0fff);
    break;

  default:
    k = sprintf (s, "erratic register values");
    break;
  }

  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;
  return l;
}


static int cs_mem (b_charptrtype s,
		   b_int32 base, b_int32 base32,
		   b_int32 limit, b_int32 limit32)
{
  int	k, l = 0;
  
  k = sprintf (s, CS_FIELD, "Memory Filter");
  INCR_BUFFER;

  switch (base & 0x000f) {
  case 0:
    /* 32-bit I/O */
    k = sprintf (s, "%08x\\h thru %08x\\h",
		 (base & 0xfff0) << 16L,
		 (limit & 0xfff0) << 16L | 0x000fffff);
    break;
    
  case 1:
    /* 64-bit I/O */
    k = sprintf (s, "%08x%08x\\h thru %08x%08x\\h",
		 base32, (base & 0xfff0) << 16L,
		 limit32, (limit & 0xfff0) << 16L | 0x000fffff);
    break;

  default:
    k = sprintf (s, "erratic register values");
    break;
  }

  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;
  return l;
}


		   
b_errtype EXPORT BestConfigScanStatusGet (
  b_handletype		handle,
  b_cscanstatustype	cscanstatus,
  b_int32	       *value
  )
{
  B_TRY_VARS_NO_PROG;
  b_int8	inbuf, outbuf [4];
  b_int16	outsize = OUT_CONFIG_SCAN_STATUS;
  
  B_TRY_BEGIN {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);
    B_TRY (Best16BitRegisterFile (handle) ? B_E_OK : B_E_NOT_E2925A);
    /* TODO: parameter check */
    inbuf = (b_int8)cscanstatus;
    B_TRY (BestBasicCommand (handle, CMD_CONFIG_SCAN_STATUS,
			     &inbuf, IN_CONFIG_SCAN_STATUS,
			     outbuf, &outsize));
    BestStream2Long (value, outbuf, 1);
  }

  B_ERRETURN (B_TRY_RET);
}


b_errtype EXPORT BestConfigScanResultGet (
  b_handletype		handle,
  b_int32      		devenum,
  b_configheadertype   *header
  )
{
  B_TRY_VARS_NO_PROG;
  b_int8	inbuf [IN_CONFIG_SCAN_RESULT];
  b_int16	outsize = OUT_CONFIG_SCAN_RESULT;
  
  B_TRY_BEGIN {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);
    B_TRY (Best16BitRegisterFile (handle) ? B_E_OK : B_E_NOT_E2925A);
    /* TODO: parameter check */
    BestLong2Stream (inbuf, &devenum, 1);
    B_TRY (BestBasicCommand (handle, CMD_CONFIG_SCAN_RESULT,
			     inbuf, IN_CONFIG_SCAN_RESULT,
			     (b_int8ptr)header, &outsize));
    BestStream2Long ((b_int32ptr)header, (b_int8ptr)header, outsize / 4);
  }

  B_ERRETURN (B_TRY_RET);
}


b_errtype EXPORT BestConfigScanResultPrint (
  b_handletype		handle,
  b_configheadertype   *header,
  b_charptrtype	       *res_string
  )
{
  int			i, k, l;  /* i: general purpose
				     k: next position where to append string
				     l: length of string */
  char		       *s;
  b_ccharptrtype       vendor, device, classcode;

  static char	buffer [2048];

  *res_string = buffer;
  s = buffer;
  l = 0;
  
  /* printing the address of the device */
  if (header -> addr & 0x00000001) {
    /* it's a config type 1 address */
    k = sprintf (s, "Bus %d, Device %x\\h, Function %d\n",
		 (header -> addr & 0x00ff0000UL) >> 16,
		 (header -> addr & 0x0000f800UL) >> 11,
		 (header -> addr & 0x00000700UL) >> 8);
    INCR_BUFFER;
  }

  else {
    /* it's a config type 0 address */
    for (i = 11; i < 32; ++i) {
      if (header -> addr & (1L << i)) {
	k = sprintf (s, "Device %x\\h (IDSEL: AD%02d), Function %d\n",
		     i - 11, i, (header -> addr & 0x00000700UL) >> 8);
	INCR_BUFFER;
	break;
      }
    }
  }

  /* print the device name */
  cs_vendor_device (header -> header.cfg_reg [0], &vendor, &device);
  k = sprintf (s, "%s, %s\n\n", vendor, device);
  INCR_BUFFER;
  
  k = cs_format (s, "Vendor ID",
		 header -> header.cfg_reg [0] & 0x0000ffff, 4);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;
  k = cs_format (s, "Device ID",
		 (header -> header.cfg_reg [0] & 0xffff0000) >> 16L, 4);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;

  k = cs_command (s, header -> header.cfg_reg [1] & 0x0000ffff);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;

  k = cs_status (s, "Status Register",
		 (header -> header.cfg_reg [1] & 0xffff0000) >> 16L);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;

  k = cs_format (s, "Revision ID",
		 header -> header.cfg_reg [2] & 0x000000ff, 2);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;
  k = cs_format (s, "Class Code",
		 (header -> header.cfg_reg [2] & 0xffffff00) >> 8L, 6);
  INCR_BUFFER;
  cs_classcode (header -> header.cfg_reg [2], &classcode);
  k = sprintf (s, "%s", classcode);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;

  k = cs_format (s, "Header Type",
		 (header -> header.cfg_reg [3] & 0x00ff0000) >> 16L, 2);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;

  k = cs_format (s, "Cache Line Size",
		 header -> header.cfg_reg [3] & 0x000000ff, 2);
  INCR_BUFFER;
  k = cs_format (s, "Interrupt Line",
		 header -> header.cfg_reg [15] & 0x000000ff, 2);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;

  k = cs_format (s, "Latency Timer",
		 (header -> header.cfg_reg [3] & 0x0000ff00) >> 8L, 2);
  INCR_BUFFER;
  k = cs_format (s, "Interrupt Pin",
		 (header -> header.cfg_reg [15] & 0x0000ff00) >> 8L, 2);
  INCR_BUFFER;
  k = cs_newline (s);
  INCR_BUFFER;

  if ((header -> header.cfg_reg [3] & 0x007f0000) == 0x00010000) {
    /* it's a bridge */
    k = cs_format (s, "Secondary Latency Timer",
		   (header -> header.cfg_reg [6] & 0xff000000) >> 24L, 2);
    INCR_BUFFER;
    k = cs_format (s, "Primary Bus",
		   header -> header.cfg_reg [6] & 0x000000ff, 2);
    INCR_BUFFER;
    k = cs_newline (s);
    INCR_BUFFER;

    k = cs_format (s, "BIST",
		   (header -> header.cfg_reg [3] & 0xff000000) >> 24L, 2);
    INCR_BUFFER;
    k = cs_format (s, "Secondary Bus",
		   (header -> header.cfg_reg [6] & 0x0000ff00) >> 8L, 2);
    INCR_BUFFER;
    k = cs_newline (s);
    INCR_BUFFER;

    k = cs_format (s, NULL, 0, 0);
    INCR_BUFFER;
    k = cs_format (s, "Subordinate Bus",
		   (header -> header.cfg_reg [6] & 0x00ff0000) >> 16L, 2);
    INCR_BUFFER;
    k = cs_newline (s);
    INCR_BUFFER;

    k = cs_bridge_control (s,  (header -> header.cfg_reg [15] & 
				0xffff0000) >> 16L);
    INCR_BUFFER;
    k = cs_newline (s);
    INCR_BUFFER;

    k = cs_status (s, "Secondary Status",
		   (header -> header.cfg_reg [7] & 0xffff0000) >> 16L);
    INCR_BUFFER;
    k = cs_newline (s);
    INCR_BUFFER;

    k = cs_baseaddr64 (s, 0, header -> header.cfg_reg [4],
		     header -> header.cfg_reg [5]);
    INCR_BUFFER;

    k = cs_mapped_io (s, header -> header.cfg_reg [8] & 0x0000ffff,
		      (header -> header.cfg_reg [8] & 0xffff0000) >> 16L);
    INCR_BUFFER;

    k = cs_io (s, header -> header.cfg_reg [7] & 0x000000ff,
	       header -> header.cfg_reg [12] & 0x0000ffff,
	       (header -> header.cfg_reg [7] & 0x0000ff00) >> 8L,
	       (header -> header.cfg_reg [12] & 0xffff0000) >> 16L);
    INCR_BUFFER;

    k = cs_mem (s, header -> header.cfg_reg [9] & 0x0000ffff,
		header -> header.cfg_reg [10],
		(header -> header.cfg_reg [9] & 0xffff0000) >> 16L,
		header -> header.cfg_reg [11]);
    INCR_BUFFER;

    k = cs_format (s, "Reserved",
		   header -> header.cfg_reg [13], 8);
    INCR_BUFFER;
    k = cs_newline (s);
    INCR_BUFFER;

    k = cs_xromaddr (s, header -> header.cfg_reg [14]);
    INCR_BUFFER;
  }

  else {
    /* it's a standard PCI device */
    k = cs_format (s, "BIST",
		   (header -> header.cfg_reg [3] & 0xff000000) >> 24L, 2);
    INCR_BUFFER;
    k = cs_format (s, "Minimum Grant",
		   (header -> header.cfg_reg [15] & 0x00ff0000) >> 16L, 2);
    INCR_BUFFER;
    k = cs_newline (s);
    INCR_BUFFER;

    k = cs_format (s, NULL, 0, 0);
    INCR_BUFFER;
    k = cs_format (s, "Maximum Latency",
		   (header -> header.cfg_reg [15] & 0xff000000) >> 24L, 2);
    INCR_BUFFER;
    k = cs_newline (s);
    INCR_BUFFER;

    k = cs_baseaddr64 (s, 0, header -> header.cfg_reg [4],
		     header -> header.cfg_reg [5]);
    INCR_BUFFER;
    k = cs_baseaddr64 (s, 2, header -> header.cfg_reg [6],
		     header -> header.cfg_reg [7]);
    INCR_BUFFER;
    k = cs_baseaddr64 (s, 4, header -> header.cfg_reg [8],
		     header -> header.cfg_reg [9]);
    INCR_BUFFER;

    k = cs_format (s, "Subsystem Vendor ID",
		   header -> header.cfg_reg [11] & 0x0000ffff, 4);
    INCR_BUFFER;
    k = cs_format (s, "Reserved 1",
		   header -> header.cfg_reg [13], 8);
    INCR_BUFFER;
    k = cs_newline (s);
    INCR_BUFFER;
    
    k = cs_format (s, "Subsystem Device ID",
		   (header -> header.cfg_reg [11] & 0xffff0000) >> 16L, 4);
    INCR_BUFFER;
    k = cs_format (s, "Reserved 2",
		   header -> header.cfg_reg [14], 8);
    INCR_BUFFER;
    k = cs_newline (s);
    INCR_BUFFER;
    
    k = cs_format (s, "CardBus CIS Pointer",
		   header -> header.cfg_reg [10], 8);
    INCR_BUFFER;
    k = cs_newline (s);
    INCR_BUFFER;

    k = cs_xromaddr (s, header -> header.cfg_reg [12]);
    INCR_BUFFER;
  }

  B_ERRETURN (B_E_OK);
}


b_errtype EXPORT BestConfigScanResultStringGet (
  b_handletype		handle,
  b_int32      		devenum,
  b_charptrtype	       *res_string
  )
{
  B_TRY_VARS_NO_PROG;
  b_configheadertype	header;

  B_TRY_BEGIN {
    B_TRY (BestConfigScanResultGet (handle, devenum, &header));
    B_TRY (BestConfigScanResultPrint (handle, &header, res_string));
  }

  B_ERRETURN (B_TRY_RET);
}


b_errtype EXPORT BestConfigScanPrint (
  b_handletype		handle,
  b_charptrtype	       *res_string
  )
{
  B_TRY_VARS_NO_PROG;
  b_int32	i;
  b_int32	devices;
  b_charptrtype	s, t;

  B_TRY_BEGIN {
    if (cscan_result [handle]) {
      free (cscan_result [handle]);
      cscan_result [handle] = NULL;
    }
    
    B_TRY (BestConfigScanStatusGet (handle, B_CSCAN_DEVICES, &devices));
    cscan_result [handle] = malloc (devices * 2048);
    B_TRY_FAIL (cscan_result [handle] ? B_E_OK : B_E_TBD);
    s = cscan_result [handle];
    *res_string = s;

    for (i = 0, s [0] = '\0'; i < devices; ++i) {
      B_TRY (BestConfigScanResultStringGet (handle, i, &t));
      strcat (s, t);
      strcat (s, "\n\n");
    }
  }

  B_ERRETURN (B_TRY_RET);
}


b_errtype EXPORT BestConfigScan (b_handletype handle)
{
  B_DECLARE_FUNCNAME ("BestConfigScan");
  B_TRY_VARS_NO_PROG;
  
  B_TRY_BEGIN {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);
    B_TRY (Best16BitRegisterFile (handle) ? B_E_OK : B_E_NOT_E2925A);
    B_TRY (BestBasicCommand (handle, CMD_CONFIG_SCAN,
			     NULL, IN_CONFIG_SCAN, NULL, NULL));
  }

  B_ERRETURN (B_TRY_RET);
}





